import React, { useEffect, useRef, useState } from "react";
import {
  Platform,
  StyleSheet,
  View,
  ActivityIndicator,
  Text,
  TouchableOpacity,
  BackHandler,
  Alert,
  StatusBar as ExpoStatusBar,
} from "react-native";
import * as SplashScreen from "expo-splash-screen";
import * as Notifications from "expo-notifications";
import * as Device from "expo-device";
import Constants from "expo-constants";
import NetInfo from "@react-native-community/netinfo";
import * as Linking from "expo-linking";
import { WebView } from "react-native-webview";
import { Audio } from "expo-av";
import * as Location from "expo-location";
import * as ScreenOrientation from "expo-screen-orientation";
import { useFonts, Vazirmatn_400Regular, Vazirmatn_700Bold } from "@expo-google-fonts/vazirmatn";
import * as WebBrowser from "expo-web-browser";
import { Ionicons } from "@expo/vector-icons";
import { Camera } from 'expo-camera';

/**
 * Developed by Native City
 * Buy this WebView source from: https://www.rtl-theme.com
 * 
 * This React Native component sets up a WebView with various configurations including offline handling,
 * push notifications, and custom user agents. This component also handles back navigation,
 * and custom JavaScript/CSS injection.
 */

// Prevent auto-hiding of splash screen until manually hidden
SplashScreen.preventAutoHideAsync();

// Notification handler configuration
Notifications.setNotificationHandler({
  handleNotification: async () => ({
    shouldShowAlert: true,
    shouldPlaySound: true,
    shouldSetBadge: false,
  }),
});

// Notification subscription type
type NotificationSubscription = {
  remove: () => void;
};

// App Permissions (Notifications, Audio, Location, Camera) - (can be null)
const appPermissions = [
  "Notifications"
];

export default function RootLayout() {
  const notificationListener = useRef<NotificationSubscription | undefined>(
    undefined
  );
  const responseListener = useRef<NotificationSubscription | undefined>(
    undefined
  );
  const [expoPushToken, setExpoPushToken] = useState<string | undefined>("");
  const [isConnected, setIsConnected] = useState(true);
  const [canGoBack, setCanGoBack] = useState(false);
  const webViewRef = useRef<WebView | null>(null);
  const offlineUrlRef = useRef<string | null>(null);
  const [canPullToRefresh, setCanPullToRefresh] = useState(false);
  const [permissionsGranted, setPermissionsGranted] = useState(false);

  // Load the custom fonts
  const [fontsLoaded] = useFonts({
    Vazirmatn_400Regular,
    Vazirmatn_700Bold,
  });

  // Base URL for the WebView  (not null)
  const [url, setUrl] = useState("https://behsabekr.ir");
  // Custom User Agent for the WebView (can be null)
  const customUserAgent = null;
  // Internal links to be opened within the WebView (not null)
  const internalLinks = ["behsabekr.ir"];
  // Deep links to be opened in the browser (can be null)
  const deepLinks = ["apparat.com"];
  // Custom JavaScript to be injected into the WebView (can be null)
  const customJS: string[] = [];
  // Custom CSS to be injected into the WebView (can be null)
  const customCSS: string[] = [];
  // Splash screen timeout configuration  (not null)
  const SPLASH_SCREEN_TIMEOUT = 2000;

  // Initialize the app: hide splash screen, register for notifications, request permissions
  useEffect(() => {
    const initializeApp = async () => {
      try {
        setTimeout(async () => {
          await SplashScreen.hideAsync();
        }, SPLASH_SCREEN_TIMEOUT);
        const permissionsGranted = await checkAndRequestPermissions();
        setPermissionsGranted(permissionsGranted);
        if (permissionsGranted) {
          const token = await registerForPushNotificationsAsync();
          if (token) {
            setExpoPushToken(token);
          }
        }
      } catch (error) {
        console.error("Error initializing app:", error);
      }
    };

    initializeApp();
  }, []);

  // Notification listeners setup
  useEffect(() => {
    if (!notificationListener.current) {
      notificationListener.current = Notifications.addNotificationReceivedListener(
        (notification) => {
          console.log(notification);
        }
      );
    }

    if (!responseListener.current) {
      responseListener.current = Notifications.addNotificationResponseReceivedListener(
        (response) => {
          console.log(response);
        }
      );
    }

    return () => {
      notificationListener.current?.remove();
      responseListener.current?.remove();
    };
  }, []);

  // Handle network connectivity changes and back button press
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsConnected(state.isConnected);
    });

    const backAction = () => {
      if (canGoBack && webViewRef.current) {
        webViewRef.current.goBack();
        return true;
      }
      return false;
    };

    const backHandler = BackHandler.addEventListener(
      "hardwareBackPress",
      backAction
    );

    return () => {
      unsubscribe();
      backHandler.remove();
    };
  }, [canGoBack]);

  // Handle reconnection after being offline
  useEffect(() => {
    if (isConnected && offlineUrlRef.current) {
      setUrl(offlineUrlRef.current);
      offlineUrlRef.current = null;
    }
  }, [isConnected]);

  // Handle WebView navigation state change
  const handleNavigationStateChange = (navState: any) => {
    setCanGoBack(navState.canGoBack);
  };

  // Decide whether to allow WebView to load a request
  const handleShouldStartLoadWithRequest = (request: any) => {
    try {
      const { url: currentUrl } = request;
      const parsedUrl = new URL(currentUrl);
      const isInternalLink = internalLinks.some((link) => {
        const httpDomain = new URL(`http://${link}`).hostname;
        const httpsDomain = new URL(`https://${link}`).hostname;
        const wwwHttpDomain = new URL(`http://www.${link}`).hostname;
        const wwwHttpsDomain = new URL(`https://${link}`).hostname;
        return (
          parsedUrl.hostname === httpDomain ||
          parsedUrl.hostname === httpsDomain ||
          parsedUrl.hostname === wwwHttpDomain ||
          parsedUrl.hostname === wwwHttpsDomain
        );
      });

      const isDeepLink = deepLinks.some((link) => {
        const httpDomain = new URL(`http://${link}`).hostname;
        const httpsDomain = new URL(`https://${link}`).hostname;
        const wwwHttpDomain = new URL(`http://www.${link}`).hostname;
        const wwwHttpsDomain = new URL(`https://${link}`).hostname;
        return (
          parsedUrl.hostname === httpDomain ||
          parsedUrl.hostname === httpsDomain ||
          parsedUrl.hostname === wwwHttpDomain ||
          parsedUrl.hostname === wwwHttpsDomain
        );
      });

      if (isDeepLink) {
        WebBrowser.openBrowserAsync(currentUrl);
        return false;
      }

      if (!isInternalLink) {
        Linking.openURL(currentUrl);
        return false;
      }

      return true;
    } catch (error) {
      return false;
    }
  };

  // Handle WebView scroll event
  const onWebViewScroll = (event: any) => {
    const { y } = event.nativeEvent.contentOffset;
    setCanPullToRefresh(y === 0);
  };

  // JavaScript to inject into the WebView for handling scroll events and fullscreen video
  const injectedJavaScript = `
    document.querySelectorAll('video').forEach(video => {
      video.addEventListener('enterpictureinpicture', function() {
        window.ReactNativeWebView.postMessage(JSON.stringify({ fullScreen: true }));
      });
      video.addEventListener('leavepictureinpicture', function() {
        window.ReactNativeWebView.postMessage(JSON.stringify({ fullScreen: false }));
      });
    });
    window.onscroll = function() {
      window.ReactNativeWebView.postMessage(JSON.stringify({scrollTop: document.documentElement.scrollTop || document.body.scrollTop}));
    };
  `;

  if (!fontsLoaded) {
    return null; // Or a fallback loading component
  }

  if (!isConnected) {
    offlineUrlRef.current = url;
    return (
      <View style={styles.offlineContainer}>
        <Ionicons name="cloud-offline-outline" size={50} color="#333" />
        <Text style={styles.offlineText}>شما آفلاین هستید</Text>
        <CustomButton
          title="تلاش دوباره"
          onPress={() => {
            NetInfo.fetch().then((state) => setIsConnected(state.isConnected));
          }}
          iconName="refresh-outline"
        />
      </View>
    );
  }

  if (!permissionsGranted) {
    return (
      <PermissionsRequestScreen
        onPermissionsGranted={() => setPermissionsGranted(true)}
      />
    );
  }

  const handleMessage = async (event: any) => {
    try {
      const messageData = JSON.parse(event.nativeEvent.data);
      if (messageData.fullScreen) {
        await ScreenOrientation.lockAsync(ScreenOrientation.OrientationLock.LANDSCAPE);
      } else if (messageData.fullScreen === false) {
        await ScreenOrientation.lockAsync(ScreenOrientation.OrientationLock.PORTRAIT);
      }

      if (messageData.scrollTop === 0) {
        setCanPullToRefresh(true);
      } else {
        setCanPullToRefresh(false);
      }
    } catch (e) {
      console.error("Failed to handle message from WebView:", e);
    }
  };

  return (
    <View style={styles.container}>
      <ExpoStatusBar translucent={true} />
      <View style={styles.customHeader} />
      <WebView
        ref={webViewRef}
        source={{ uri: url }}
        style={styles.webview}
        {...(customUserAgent && { userAgent: customUserAgent })}
        onShouldStartLoadWithRequest={handleShouldStartLoadWithRequest}
        onNavigationStateChange={handleNavigationStateChange}
        injectedJavaScript={`
          document.querySelectorAll('a[target="_blank"]').forEach(function(anchor) {
            anchor.target = "_self";
          });
          ${injectedJavaScript}
          ${customJS.join("\n")}
        `}
        injectedJavaScriptBeforeContentLoaded={`
          ${customCSS.map((css) => `<style>${css}</style>`).join("\n")}
        `}
        renderLoading={() => (
          <View style={styles.loaderContainer}>
            <ActivityIndicator color="#009b88" size="large" />
            <Text style={styles.loadingText}>در حال بارگذاری...</Text>
          </View>
        )}
        startInLoadingState={true}
        allowsInlineMediaPlayback
        javaScriptEnabled
        domStorageEnabled
        onMessage={handleMessage}
        allowsFullscreenVideo={true}
        mediaPlaybackRequiresUserAction={false}
        onError={(e) => console.error("WebView error:", e)}
        onLoadEnd={() => console.log("WebView finished loading")}
        onScroll={onWebViewScroll}
      />
    </View>
  );
}

// Custom Button component with optional Icon
const CustomButton = ({ title, onPress, iconName }: {
  title: string;
  onPress: () => void;
  iconName?: keyof typeof Ionicons.glyphMap;
}) => {
  return (
    <TouchableOpacity onPress={onPress} style={styles.buttonContainer}>
      {iconName && (
        <Ionicons name={iconName} size={20} color="#fff" style={styles.buttonIcon} />
      )}
      <Text style={styles.buttonText}>{title}</Text>
    </TouchableOpacity>
  );
};

// Component to request permissions with a modern UI
const PermissionsRequestScreen = ({
  onPermissionsGranted,
}: {
  onPermissionsGranted: () => void;
}) => {
  const requestPermissions = async () => {
    const permissionsGranted = await checkAndRequestPermissions();
    if (permissionsGranted) {
      onPermissionsGranted();
    }
  };

  const openSettings = async () => {
    try {
      await Linking.openSettings();
    } catch (error) {
      Alert.alert('خطای دسترسی', "خطایی رخ داد. لطفاً به صورت دستی وارد تنظیمات شوید.", [
        {
          text: 'باشه',
          style: 'cancel',
        },
      ]);
    }
  };

  return (
    <View style={styles.permissionsContainer}>
      <Ionicons name="lock-closed-outline" size={50} color="#333" />
      <Text style={styles.permissionsTitle}>مجوزهای مورد نیاز</Text>
      <Text style={styles.permissionsText}>
        برای استفاده از این برنامه، ما به مجوزهای زیر نیاز داریم:
      </Text>
      {appPermissions.includes("Audio") && (
        <Text style={styles.permissionsText}>- میکروفون</Text>
      )}
      {appPermissions.includes("Location") && (
        <Text style={styles.permissionsText}>- موقعیت مکانی</Text>
      )}
      {appPermissions.includes("Notifications") && (
        <Text style={styles.permissionsText}>- اعلان‌ها</Text>
      )}
      {appPermissions.includes("Camera") && (
        <Text style={styles.permissionsText}>- دوربین</Text>
      )}
      <CustomButton title="فعال کردن مجوزها" onPress={requestPermissions} iconName="checkmark-circle-outline" />
      <CustomButton title="باز کردن تنظیمات" onPress={openSettings} iconName="settings-outline" />
      <Text style={styles.privacyText}>
        ما به حریم خصوصی شما احترام می‌گذاریم و از این مجوزها تنها برای بهبود
        تجربه کاربری شما استفاده می‌کنیم. اگر گزینه "فعال کردن مجوزها" کار
        نکرد، دسترسی ها را از تنظیمات برنامه فعال کنید.
      </Text>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: "#fff",
  },
  customHeader: {
    height: 33,
    backgroundColor: "#000",
  },
  webview: {
    flex: 1,
  },
  offlineContainer: {
    flex: 1,
    justifyContent: "center",
    alignItems: "center",
    backgroundColor: "#f8f9fa",
  },
  offlineText: {
    fontSize: 20,
    marginBottom: 20,
    color: "#333",
    fontFamily: "Vazirmatn_400Regular",
  },
  loaderContainer: {
    ...StyleSheet.absoluteFillObject,
    justifyContent: "center",
    alignItems: "center",
    backgroundColor: "rgba(255, 255, 255, 0.8)",
  },
  loadingText: {
    fontSize: 16,
    marginTop: 10,
    color: "#333",
    fontFamily: "Vazirmatn_400Regular",
  },
  permissionsContainer: {
    flex: 1,
    justifyContent: "center",
    alignItems: "center",
    padding: 20,
    backgroundColor: "#f8f9fa",
  },
  permissionsTitle: {
    fontSize: 24,
    marginBottom: 20,
    color: "#333",
    fontFamily: "Vazirmatn_400Regular",
  },
  permissionsText: {
    fontSize: 18,
    marginBottom: 10,
    color: "#555",
    fontFamily: "Vazirmatn_400Regular",
  },
  privacyText: {
    marginTop: 20,
    fontSize: 14,
    color: "#777",
    textAlign: "center",
    fontFamily: "Vazirmatn_400Regular",
  },
  buttonContainer: {
    backgroundColor: "#009b88",
    padding: 10,
    borderRadius: 5,
    marginTop: 10,
    flexDirection: "row",
    alignItems: "center",
    justifyContent: "center",
  },
  buttonIcon: {
    marginRight: 10,
  },
  buttonText: {
    fontFamily: "Vazirmatn_700Bold",
    fontSize: 16,
    color: "#fff",
    textAlign: "center",
  },
});

// Register for push notifications and get the token
async function registerForPushNotificationsAsync() {
  let token;

  if (Platform.OS === "android") {
    await Notifications.setNotificationChannelAsync("default", {
      name: "default",
      importance: Notifications.AndroidImportance.MAX,
      vibrationPattern: [0, 250, 250, 250],
      lightColor: "#FF231F7C",
    });
  }

  if (Device.isDevice) {
    try {
      const projectId =
        Constants?.expoConfig?.extra?.eas?.projectId ??
        Constants?.easConfig?.projectId;
      if (!projectId) {
        throw new Error("Project ID not found");
      }
      token = (
        await Notifications.getExpoPushTokenAsync({ projectId })
      ).data;
    } catch (e) {
      console.error(e);
    }
  } else {
    alert("Must use a physical device for Push Notifications");
  }

  return token;
}

// Check and request necessary permissions
async function checkAndRequestPermissions() {
  let permissionsGranted = true;

  if (appPermissions.includes("Audio")) {
    const { status: microphoneStatus } = await Audio.requestPermissionsAsync();
    if (microphoneStatus !== "granted") {
      Alert.alert('خطای دسترسی', "متاسفیم، ما به مجوز میکروفون نیاز داریم!\nاین دسترسی برای ورود به برنامه اجباری است.", [
        {
          text: 'باشه',
          style: 'cancel',
        },
      ]);
      permissionsGranted = false;
    }
  }

  if (appPermissions.includes("Location")) {
    const { status: locationStatus } =
      await Location.requestForegroundPermissionsAsync();
    if (locationStatus !== "granted") {
      Alert.alert('خطای دسترسی', "متاسفیم، ما به مجوز موقعیت مکانی نیاز داریم!\nاین دسترسی برای ورود به برنامه اجباری است.", [
        {
          text: 'باشه',
          style: 'cancel',
        },
      ]);
      permissionsGranted = false;
    }
  }

  if (appPermissions.includes("Camera")) {
    const { status: cameraStatus } = await Camera.requestCameraPermissionsAsync();
    if (cameraStatus !== "granted") {
      Alert.alert('خطای دسترسی', "متاسفیم، ما به مجوز دوربین نیاز داریم!\nاین دسترسی برای ورود به برنامه اجباری است.", [
        {
          text: 'باشه',
          style: 'cancel',
        },
      ]);
      permissionsGranted = false;
    }
  }

  if (appPermissions.includes("Notifications")) {
    const { status: existingStatus } = await Notifications.getPermissionsAsync();
    let finalStatus = existingStatus;
    if (finalStatus !== "granted") {
      const { status } = await Notifications.requestPermissionsAsync();
      finalStatus = status;
    }

    if (finalStatus !== "granted") {
      Alert.alert('خطای دسترسی', "متاسفیم، ما به مجوز اعلان ها نیاز داریم!\nاین دسترسی برای ورود به برنامه اجباری است.", [
        {
          text: 'باشه',
          style: 'cancel',
        },
      ]);
      permissionsGranted = false;
    }
  }

  return permissionsGranted;
}